<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\AusBank;
use App\Models\CanBank;
use App\Models\Country;
use App\Models\UkBanks;
use App\Models\UsBanks;
use App\Models\Currency;
use App\Models\Transaction;
use App\Models\Notification;
use App\Models\TransferCode;
use App\Models\WireTransfer;
use Illuminate\Http\Request;
use App\Helpers\TransferHelper;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class WireTransferController extends Controller
{

    public function getBanks(Request $request)
    {
        $countryName = $request->input('country_name');

        // Get banks based on selected country
        $banks = match ($countryName) {
            'United Kingdom' => UkBanks::select('name')->get(),
            'United States' => UsBanks::select('name')->get(),
            'Australia' => AusBank::select('name')->get(),
            'Canada' => CanBank::select('name')->get(),
            default => collect()
                ->merge(UkBanks::select('name')->get()->toArray())
                ->merge(UsBanks::select('name')->get()->toArray())
                ->merge(AusBank::select('name')->get()->toArray())
                ->merge(CanBank::select('name')->get()->toArray()),
        };

        return response()->json(['banks' => $banks]);
    }



    public function startwire(Request $request)
    {
        // dd($request->all());
        $user = Auth::user(); // Get the authenticated user

        $amount = $request->input('amount');
        if ($user->account_balance < $amount) {
            return back()->withErrors(['error' => 'Insufficient funds.']);
        }

        // Calculate total transfers for the current month using the helper
        $totalTransfersForMonth = TransferHelper::getTotalTransfersForMonth($user->id);

        // Check if the new transfer exceeds the user's monthly limit
        if (($totalTransfersForMonth + $request->amount) > $user->account_limit) {
            return redirect()->back()->with([
                'error' => 'You have exceeded Your transfer limit for the month!',
            ]);
        }


        // Create the wire transfer but don't process the balance yet
        $save = new WireTransfer;
        $save->user_id = $user->id; // Get the authenticated user's ID
        $save->beneficiary_bank_country = trim($request->beneficiary_bank_country);
        $save->beneficiary_bank_name = trim($request->beneficiary_bank_name);
        $save->beneficiary_account_type = trim($request->beneficiary_account_type);
        $save->beneficiary_swift_code = trim($request->beneficiary_swift_code);
        $save->beneficiary_name = trim($request->beneficiary_name);
        $save->beneficiary_account_number = trim($request->beneficiary_account_number);
        $save->currency = trim($request->currency);
        $save->amount = trim($request->amount);
        $save->description = trim($request->description);
        $save->transaction_reference = str_pad(mt_rand(0, 999999999999), 20, '0', STR_PAD_LEFT);
        $save->status = 'pending'; // Mark the status as pending initially
        $save->save();

        // Store the wire transfer ID in the session for code validation
        session(['wire_transfer_id' => $save->id]);

        // Redirect to COT code input page
        return redirect('account/code/cot');
    }

    public function cotCodePage()
    {
        $cotCode = TransferCode::where('name', 'COT Code')->where('is_enabled', true)->first();
        if (!$cotCode) {
            return redirect()->route('imf.code.page'); // Skip to IMF code if COT code is not enabled
        }

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();
        return view('account.code.cot', compact('notifications')); // Page for entering COT code
    }

    public function validateCotCode(Request $request)
    {
        $cotCode = TransferCode::where('name', 'COT Code')->where('is_enabled', true)->first();
        if ($cotCode && $request->transfer_code != $cotCode->code) {
            return redirect()->back()->with('error', 'Invalid COT Code.');
        }

        // If COT code is correct or skipped, proceed to IMF code
        session(['cot_code_passed' => true]);
        return redirect()->route('imf.code.page');
    }

    public function imfCodePage()
    {
        $imfCode = TransferCode::where('name', 'IMF Code')->where('is_enabled', true)->first();
        if (!$imfCode) {
            return redirect()->route('tax.code.page'); // Skip to Tax code if IMF code is not enabled
        }

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();

        return view('account.code.imf', compact('notifications')); // Page for entering IMF code
    }

    public function validateImfCode(Request $request)
    {
        $imfCode = TransferCode::where('name', 'IMF Code')->where('is_enabled', true)->first();
        if ($imfCode && $request->transfer_code != $imfCode->code) {
            return redirect()->back()->with('error', 'Invalid IMF Code.');
        }

        // If IMF code is correct or skipped, proceed to Tax code
        session(['imf_code_passed' => true]);
        return redirect()->route('tax.code.page');
    }

    public function taxCodePage()
    {
        $taxCode = TransferCode::where('name', 'Tax Code')->where('is_enabled', true)->first();
        if (!$taxCode) {
            return redirect()->route('complete.wire.transfer'); // Skip to completion if Tax code is not enabled
        }

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();

        return view('account.code.tax', compact('notifications')); // Page for entering Tax code
    }

    public function validateTaxCode(Request $request)
    {
        $taxCode = TransferCode::where('name', 'Tax Code')->where('is_enabled', true)->first();
        if ($taxCode && $request->transfer_code != $taxCode->code) {
            return redirect()->back()->with('error', 'Invalid Tax Code.');
        }

        // If Tax code is correct or skipped, complete the wire transfer
        session(['tax_code_passed' => true]);
        return redirect()->route('complete.wire.transfer');
    }

    public function completeWireTransfer()
    {
        $user = Auth::user(); // Get the authenticated user

        // Fetch the required codes from the transfer_code table
        $cotCodeRequired = DB::table('transfer_codes')->where('name', 'COT Code')->value('is_enabled');
        $imfCodeRequired = DB::table('transfer_codes')->where('name', 'IMF Code')->value('is_enabled');
        $taxCodeRequired = DB::table('transfer_codes')->where('name', 'Tax Code')->value('is_enabled');

        // Ensure that required codes have been validated before proceeding
        if (($cotCodeRequired && !session('cot_code_passed')) ||
            ($imfCodeRequired && !session('imf_code_passed')) ||
            ($taxCodeRequired && !session('tax_code_passed'))
        ) {

            // If a required code is not validated, return an error
            return redirect()->back()->with('error', 'Transfer process incomplete or code validation required.');
        }

        // Proceed with the transfer if the required codes are validated or skipped
        $wireTransfer = WireTransfer::find(session('wire_transfer_id'));

        if (!$wireTransfer) {
            return redirect()->back()->with('error', 'Wire Transfer not found.');
        }

        // Fetch the wire transfer fee from website_settings
        $wireFee = DB::table('website_settings')->value('wire_fee');
        $wireFee = $wireFee ? (float) $wireFee : 0;

        // Deduct the transfer amount from the user's balance
        $user->account_balance -= $wireTransfer->amount;
        $user->save();

        // Update the wire transfer status to 'completed'
        $wireTransfer->status = 'Pending';
        $wireTransfer->transaction_type = 'Wire Transfer';
        $wireTransfer->fee = $wireFee;
        $wireTransfer->save();

        $transactionReference = $wireTransfer->transaction_reference;

        // Log the transaction in the transactions table
        $transactionHistory = new Transaction;
        $transactionHistory->user_id = $user->id;
        $transactionHistory->transaction_type = 'Wire Transfer';
        $transactionHistory->amount  = $wireTransfer['amount']; // Store transfer amount + fee
        $transactionHistory->account_name = $wireTransfer['beneficiary_name'];
        $transactionHistory->account_number = $wireTransfer['beneficiary_account_number'];
        $transactionHistory->bank_name = $wireTransfer['beneficiary_bank_name'];
        $transactionHistory->description = $wireTransfer['description'];
        $transactionHistory->fee = $wireFee;
        $transactionHistory->status = 'Pending'; // Set status to 'Pending' until fully processed
        $transactionHistory->currency = $wireTransfer['currency']; // Set status to 'Pending' until fully processed
        $transactionHistory->reference = $transactionReference;
        $transactionHistory->save();

        // Send an email notification to the user
        Mail::to($user->email)->send(new \App\Mail\WireTransferNotification($wireTransfer));

        // Trigger a database notification
        Notification::create([
            'user_id' => $user->id,
            'type' => 'Wire Transfer',
            'data' => 'You have successfully initiated a wire transfer of $' . $wireTransfer->amount .
                ' to ' . $wireTransfer->beneficiary_name . '. Status: Pending.',
            'is_read' => false, // Unread by default
        ]);

        // Clear the session codes and wire transfer ID
        session()->forget(['cot_code_passed', 'imf_code_passed', 'tax_code_passed', 'wire_transfer_id']);

        return redirect()->route('transfer.success', ['transaction_reference' => $transactionReference])->with([
            'amount' => $wireTransfer->amount,
            'status' => $wireTransfer->status,
            'beneficiary_name' => $wireTransfer->beneficiary_name,
            'transaction_reference' => $wireTransfer->transaction_reference,
            'bank_name' => $wireTransfer->beneficiary_bank_name,
            'description' => $wireTransfer->description,
            'message' => 'Wire Transfer completed successfully.'
        ]);
    }


    public function showSuccessPage($transaction_reference)
    {
        // Try to fetch from wire_transfers table
        $transfer = Transaction::where('reference', $transaction_reference)->first();

        // Retrieve unread notifications for the user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('account.transfer_success', compact('transfer', 'notifications'));
    }


    public function account_intl()
    {
        // Fetch banks from both usbanks and ukbanks tables
        $usbanks = DB::table('us_banks')->select('id', 'name')->get();
        $ukbanks = DB::table('uk_banks')->select('id', 'name')->get();

        // Combine the two collections
        $banks = $usbanks->merge($ukbanks);

        $data['countries'] = Country::where('status', true)->get();
        $data['currency'] = Currency::where('status', true)->get();

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();

        return view('account.intl', $data, compact('banks', 'notifications'));
    }


    public function admintranswire(Request $request)
    {
        $wire = WireTransfer::latest()->paginate(10)->appends(request()->query());
        return view('admin/transactions/wire', compact('wire'));
    }


    public function adminwireadd(Request $request)
    {
        // Fetch banks from both usbanks and ukbanks tables
        $usbanks = DB::table('us_banks')->select('id', 'name')->get();
        $ukbanks = DB::table('uk_banks')->select('id', 'name')->get();
        $canbank = DB::table('can_banks')->select('id', 'name')->get();
        $ausbank = DB::table('aus_banks')->select('id', 'name')->get();

        // Merge all collections
        $banks = $usbanks->merge($ukbanks)->merge($canbank)->merge($ausbank);

        $data['user'] = User::where('role', 'user')->get();
        $data['currency'] = Currency::where('status', true)->get();
        $country = Country::where('status', true)->get();

        return view('admin/transactions/wire_add', $data, compact('banks', 'country'));
    }


    public function adminwirehistoryadd(Request $request)
    {
        //   dd($request->all());
        $wireFee = DB::table('website_settings')->value('wire_fee');
        $wireFee = $wireFee ? (float) $wireFee : 0;

        // Create a new transaction using mass assignment
        $user = User::find(Auth::user()->id);
        $user = new WireTransfer;
        $user->user_id     = trim($request->user_id);
        $user->amount = trim($request->amount);
        $user->beneficiary_bank_name = trim($request->beneficiary_bank_name);
        $user->beneficiary_bank_country = trim($request->beneficiary_bank_country);
        $user->beneficiary_name = trim($request->beneficiary_name);
        $user->beneficiary_account_number = trim($request->beneficiary_account_number);
        $user->beneficiary_account_type = trim($request->beneficiary_account_type);
        $user->beneficiary_swift_code = trim($request->beneficiary_swift_code);
        $user->fee = $wireFee;
        $user->currency = trim($request->currency);
        $user->status    = trim($request->status);
        $user->transaction_type    = trim($request->transaction_type);
        $user->description    = trim($request->description);
        $user->created_at    = trim($request->created_at);
        $user->completed_at    = trim($request->completed_at);
        $user->transaction_reference = str_pad(mt_rand(0, 999999999999), 20, '0', STR_PAD_LEFT);
        $user->save();

        return redirect('admin/transactions/wire')->with('message', 'Wire history added successfully.');
    }


    public function viewwireReceipt($id)
    {
        // Fetch the transfer details
        $transfer = WireTransfer::findOrFail($id);

        // Format the transfer date to '26 Nov 2024 06:45 PM'
        $transfer->formatted_date = \Carbon\Carbon::parse($transfer->initiated_at)->format('d M Y h:i A');

        // Generate the PDF using a Blade template
        $pdf = Pdf::loadView('account/wire_receipt', compact('transfer'));

        // Return the PDF as a download
        return $pdf->download('wire_receipt_' . $transfer->id . '.pdf');
    }
}
