<?php

namespace App\Http\Controllers;

use Str;
use App\Models\Transaction;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Models\LocalTransfer;
use App\Helpers\TransferHelper;
use App\Models\WebsiteSettings;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\DomesticTransferNotification;
use Illuminate\Support\Facades\Log;

class LocalTransferController extends Controller
{

    public function accounttransfer(Request $request)
    {
        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
                                    ->where('is_read', false)  // Filter by unread notifications
                                    ->orderBy('created_at', 'desc')
                                    ->get();
        return view('account/transfer', compact('notifications'));
    }

    
    public function account_local()
    {
         // Fetch banks from both usbanks and ukbanks tables
         $usbanks = DB::table('us_banks')->select('id', 'name')->get();
         $ukbanks = DB::table('uk_banks')->select('id', 'name')->get();
  
         // Combine the two collections
         $banks = $usbanks->merge($ukbanks);

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
                                    ->where('is_read', false)  // Filter by unread notifications
                                    ->orderBy('created_at', 'desc')
                                    ->get();
        return view('account.local',compact('banks', 'notifications'));
    }

    public function submitTransfer(Request $request)
    {
      //  dd($request->all());
        // Validate the input
        $request->validate([
            'beneficiary_name' => 'required|string|max:255',
            'beneficiary_bank_name' => 'required|string|max:255',
            'beneficiary_account_number' => 'required|string|max:20',
            'beneficiary_account_type' => 'required|in:Savings Account,Current Account,Retirement Account,Money Market Account (MMAs),Checking Account,Domiciliary Account,Online Banking',
            'amount' => 'required|numeric|min:0.01',
            'description' => 'nullable|string',
        ]);

        $user = Auth::user();
        $amount = $request->input('amount');
        if ($user->account_balance < $amount) {
            return back()->withErrors(['error' => 'Insufficient funds.']);
        }

        // Calculate total transfers for the current month using the helper
        $totalTransfersForMonth = TransferHelper::getTotalTransfersForMonth($user->id);

        // Check if the new transfer exceeds the user's monthly limit
        if (($totalTransfersForMonth + $request->amount) > $user->account_limit) {
            return redirect()->back()->with([
                'error' => 'You have exceeded Your transfer limit for the month!',
            ]);
        }

        // Store transfer details in session temporarily
        session([
            'transfer_data' => $request->all(),
            'transaction_reference' => str_pad(mt_rand(0, 999999999999), 12, '0', STR_PAD_LEFT),
        ]);

        // Redirect user to the transfer pin input form
        return redirect()->route('account.transfer_pin');
    }

    public function requestPin()
    {
        // Retrieve transfer data from session
        $transferData = session('transfer_data');

        // Check if session data exists
        if (!$transferData) {
            return redirect()->route('account.dashboard')->with('error', 'Transfer session expired.');
        }

        return view('account.transfer_pin',compact('transferData'));
    }


    public function verifyPin(Request $request)
    {
        // Validate the pin
        $request->validate([
            'acct_pin' => 'required|numeric',
        ]);

        // Get the authenticated user
        $user = Auth::user();

        // Verify the pin
        if ($user->acct_pin !== $request->acct_pin) {
            return back()->withErrors(['acct_pin' => 'Invalid transfer pin.']);
        }

        // Retrieve the stored transfer data
        $transferData = session('transfer_data');
        $transactionReference = session('transaction_reference');

        // Fetch the domestic transfer fee from website_settings
        $domesticFee = DB::table('website_settings')->value('domestic_fee');
        $domesticFee = $domesticFee ? (float) $domesticFee : 0;
        
        // Calculate the total amount to be deducted (transfer amount + fee)
        $totalAmount = $transferData['amount'] + $domesticFee;

        // Check if the user has enough balance
        if ($user->account_balance < $transferData['amount']) {
            return back()->withErrors(['error' => 'Insufficient funds including transfer fee.']);
        }

        // Create the local transfer
        $localTransfer = new LocalTransfer;
        $localTransfer->user_id = $user->id;
        $localTransfer->beneficiary_name = $transferData['beneficiary_name'];
        $localTransfer->beneficiary_bank_name = $transferData['beneficiary_bank_name'];
        $localTransfer->beneficiary_account_number = $transferData['beneficiary_account_number'];
        $localTransfer->beneficiary_account_type = $transferData['beneficiary_account_type'];
        $localTransfer->amount = $transferData['amount'];
        $localTransfer->description = $transferData['description'] ?? null;
        $localTransfer->transaction_reference = $transactionReference;
        $localTransfer->status = 'Pending'; // Initially set status to pending
        $localTransfer->initiated_at = now();
        $localTransfer->fee = $domesticFee;
        $localTransfer->transaction_type = 'Outward Transfer';
        $localTransfer->save(); // Save the transfer
    
        // Deduct the total amount from the user's balance
        $user->account_balance -= $totalAmount;
        $user->save();

        $currency = $user->acct_currency;

        // Log the transaction in the transactions table
        $transactionHistory = new Transaction;
        $transactionHistory->user_id = $user->id;
        $transactionHistory->transaction_type = 'Domestic Transfer';
        $transactionHistory->amount  = $transferData['amount']; // Store transfer amount + fee
        $transactionHistory->account_name = $transferData['beneficiary_name'];
        $transactionHistory->account_number = $transferData['beneficiary_account_number'];
        $transactionHistory->bank_name = $transferData['beneficiary_bank_name'];
        $transactionHistory->description = $transferData['description'];
        $transactionHistory->fee = $domesticFee;
        $transactionHistory->status = 'Pending'; // Set status to 'Pending' until fully processed
        $transactionHistory->currency = $currency; // Set status to 'Pending' until fully processed
        $transactionHistory->reference = $transactionReference;
        $transactionHistory->save();

         // Send email to the user
        Mail::to($user->email)->send(new DomesticTransferNotification($localTransfer));
        
        // Trigger a notification for the user
        Notification::create([
            'user_id' => $user->id,
            'type' => 'Domestic Transfer',
            'data' => 'You have successfully initiated a domestic transfer of $' . $transferData['amount'] . 
                    ' to ' . $transferData['beneficiary_name'] . '. Status: Pending.',
            'is_read' => false, // Unread by default
        ]);

        // Clear session data
        session()->forget(['transfer_data', 'transaction_reference']);

        // Redirect or show success message
        return redirect()->route('domestic_transfer.success', ['transaction_reference' => $transactionReference]);
    }


    public function domesticSuccess($transaction_reference)
    {
        // Fetch the transfer details
        $transfer = Transaction::where('reference', $transaction_reference)->firstOrFail();

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
                                    ->where('is_read', false)  // Filter by unread notifications
                                    ->orderBy('created_at', 'desc')
                                    ->get();
        return view('account/domestic_success', compact('transfer', 'notifications'));
    }



}
