<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Loans;
use App\Models\InterestRate;
use App\Models\LoanRepayment;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoanController extends Controller
{

    public function account_getloan(Request $request)
    {
        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();
        $loan = InterestRate::all();
        return view('account.getloan', compact('notifications', 'loan'));
    }


    public function getInterestRate($loan_type, $loan_term)
    {
        $interest = InterestRate::where('loan_type', $loan_type)
            ->where('min_loan_term', '<=', $loan_term)
            ->where('max_loan_term', '>=', $loan_term)
            ->first();

        if ($interest) {
            return response()->json(['interest_rate' => $interest->interest_rate]);
        } else {
            return response()->json(['interest_rate' => 'Not available']);
        }
    }


    // User applies for a loan
    public function applyForLoan(Request $request)
    {
        $user = Auth::user();

        // Check if loan amount exceeds the limit
        // Get the total amount of approved loans for the user
        $approvedLoansTotal = Loans::where('user_id', $user->id)
            ->where('status', 'Approved')
            ->sum('loan_amount'); // Sum the loan_amount of all approved loans

        // Check if the total approved loans exceed 200,000
        $loanAmount = trim($request->loan_amount);
        if (($approvedLoansTotal + $loanAmount) > 200000) {
            return redirect()->back()->with('error', 'You have exceeded your maximum loan application amount of $200,000. Please Pay up your outstanding');
        }

        // Create a new loan application
        $loan = new Loans();
        $loan->user_id = Auth::id(); // Get the currently authenticated user ID
        $loan->loan_type = trim($request->loan_type);
        $loan->loan_amount = trim($request->loan_amount);
        $loan->interest_rate = trim($request->interest_rate);
        $loan->loan_term = trim($request->loan_term);
        $loan->balance = trim($request->balance); // Initially the balance is the full loan amount
        $loan->status = 'Pending'; // Set status to Pending until admin approves
        $loan->save();

        Notification::create([
            'user_id' => Auth::id(),  // Use Auth to get the logged-in user's ID
            'type' => 'Loan Application',  // Type of the notification
            'data' => 'You have successfully submitted a loan application of $' . number_format($loan->loan_amount, 2) .
                ' for ' . $loan->loan_type . ' loan Status: Pending.',
            'is_read' => false,  // Notification is unread by default
        ]);

        // Return response
        return redirect('account/loan')->with(['message' => 'Loan application submitted successfully! Waiting for admin approval.']);
    }




    public function getrepayLoan(Request $request, $id)
    {
        $loan = Loans::findOrFail($id);
        $notifications = Notification::where('user_id', Auth::id())
            ->where('is_read', false)  // Filter by unread notifications
            ->orderBy('created_at', 'desc')
            ->get();
        return view('account.loan_repay', compact('loan', 'notifications'));
    }


    public function repayLoan(Request $request, $id, $userId)
    {
        $loan = Loans::findOrFail($id);

        if ($loan->status != 'Approved' || $loan->balance <= 0) {
            return redirect()->back()->with('error', 'Loan is not available for repayment.');
        }

        $request->validate([
            'payment_amount' => 'required|numeric|min:1|max:' . $loan->balance,
        ]);

        $paymentAmount = floatval($request->input('payment_amount'));

        // Record repayment
        LoanRepayment::create([
            'loan_id' => $loan->id,
            'payment_amount' => $paymentAmount,
            'payment_date' => now(),
        ]);

        // Update loan balance
        $loan->balance -= $paymentAmount;
        $loan->save();

        // Get the authenticated user
        $user = Auth::user();

        // Deduct from user's loan_balance and account_balance
        $user->loan_balance = max(0, (float) $user->loan_balance - $paymentAmount);
        $user->account_balance = max(0, (float) $user->account_balance - $paymentAmount);
        $user->save();

        // Send appropriate notification
        if ($loan->balance <= 0) {
            $loan->status = 'Paid';
            $loan->save();

            Notification::create([
                'user_id' => $user->id,
                'type' => 'Loan Repayment',
                'data' => '🎉 Congratulations! Your loan of $' . number_format($loan->loan_amount, 2) . ' has been fully paid off.',
                'is_read' => false,
            ]);
        } else {
            Notification::create([
                'user_id' => $user->id,
                'type' => 'Loan Repayment',
                'data' => '✅ You repaid $' . number_format($paymentAmount, 2) . ' toward your loan. Remaining balance: $' . number_format($loan->balance, 2),
                'is_read' => false,
            ]);
        }

        return redirect()->route('account.loan', $id)->with('success', 'Payment made successfully.');
    }


    public function adminloanlist()
    {
        $data['loans'] = Loans::all();
        return view('admin.loans.list', $data);
    }


    // Admin approves or rejects a loan
    public function updateLoanStatus(Request $request, $loanId)
    {
        // Validate the request for status update
        $request->validate([
            'status' => 'required|in:Approved,Rejected',
        ]);

        // Find the loan by ID
        $loan = Loans::findOrFail($loanId);

        // Find the user associated with the loan
        $user = $loan->user;

        // Update loan status
        $loan->status = $request->status;

        if ($loan->status === 'Approved') {
            // Loan is approved, add loan amount to the user's balance
            $user->loan_balance += $loan->loan_amount;

            // Send notification to user about loan approval
            Notification::create([
                'user_id' => $user->id,
                'type' => 'Loan Status Update',
                'data' => 'Congratulations! Your loan application for $' . number_format($loan->loan_amount, 2) . ' has been approved. Funds sent to your loan Balance.',
                'is_read' => false,  // Mark the notification as unread by default
            ]);

            // Save the updated user balance
            $user->save();
        } elseif ($loan->status === 'Rejected') {
            // Loan is rejected, reset the balance to 0
            $loan->balance = 0;

            // Send notification to user about loan rejection
            Notification::create([
                'user_id' => $user->id,
                'type' => 'Loan Status Update',
                'data' => 'We regret to inform you that your loan application for $' . number_format($loan->loan_amount, 2) . ' has been rejected.',
                'is_read' => false,  // Mark the notification as unread by default
            ]);
        }

        // Save the loan status update
        $loan->save();

        // Return response or redirect back to admin loan list
        return redirect('admin/loans/list')->with(['message' => 'Loan status updated successfully!']);
    }
}
