<?php

namespace App\Http\Controllers;

use App\Models\Cards;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Models\CardTransactionHistory;
use App\Mail\CardApplicationNotification;

class CardController extends Controller
{
    public function usercardlist(Request $request)
    {
        // Get the authenticated user
        $user = Auth::user();

        // Count the number of cards with 'Pending' status for the authenticated user
        $data['card'] = Cards::where('user_id', $user->id)
            ->where('card_status', 'Pending') // Assuming 'Pending' is the status for pending transactions
            ->count();

        // Retrieve all cards for the authenticated user
        $data['cards'] = Cards::where('user_id', $user->id)->get();

        $card = Cards::where('user_id', auth()->id())->first(); // Get the first card of the authenticated user

        if ($card) {
            $cardhistory = CardTransactionHistory::where('card_id', $card->id)
                                                ->latest()
                                                ->take(10)
                                                ->get();
        } else {
            $cardhistory = collect(); // Return an empty collection if the user doesn't have a card
        }

        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
                ->where('is_read', false)  // Filter by unread notifications
                ->orderBy('created_at', 'desc')
                ->get();

        return view('account.cards', $data, compact('cardhistory', 'notifications'));
    }


    public function createcard(Request $request)
    {
        // Retrieve only unread notifications (where is_read is false) for the authenticated user
        $notifications = Notification::where('user_id', Auth::id())
                ->where('is_read', false)  // Filter by unread notifications
                ->orderBy('created_at', 'desc')
                ->get();
        return view('account.cards_apply', compact('notifications'));
    }


    public function cardstore(Request $request)
    {
        $request->validate([
            'card_type' => 'required|in:Debit,Credit',
            'card_brand' => 'required|in:Visa,Mastercard,Amex,Discover',
            'card_pin' => 'required|string|min:4|max:4',
        ]);

        // Retrieve the card application fee from the website_settings table
        $cardFee = DB::table('website_settings')->value('card_fee');

        // Get the authenticated user
        $user = auth()->user();

        // 🔒 Check if the user already has a card
        $existingCard = Cards::where('user_id', $user->id)->first();
        if ($existingCard) {
            return redirect()->back()->with('error', 'You already have a Virtual card. You cannot apply for more than one.');
        }

        // Check if the user has enough balance to cover the card fee
        if ($user->account_balance < $cardFee) {
            return redirect()->back()->with('error', 'Insufficient balance to apply for a card.');
        }

        // Subtract the card fee from the user's balance
        $user->account_balance -= $cardFee;
        $user->save();

        // Create the card for the user
        $card = new Cards();
        $card->user_id = $user->id;
        $card->card_type = $request->input('card_type');
        $card->card_brand = $request->input('card_brand');
        
        // Automatically set the expiry date to 5 years ahead
        $card->expiry_date = now()->addYears(5);

        // Automatically generate the CVV based on the card brand
        $card->cvv = $this->generateCvv($request->input('card_brand'));

        $card->card_pin = $request->input('card_pin');
        $card->card_status = 'Pending';
        $card->issued_date = now();
        $card->balance = '0';
        $card->card_number = $this->generateCardNumber($request->input('card_brand')); // Pass the card brand
        $card->save();


        // Send an email notification to the user
        Mail::to($user->email)->send(new CardApplicationNotification($card));

        // Create an in-app notification for the user
        Notification::create([
            'user_id' => $user->id,
            'type' => 'Card Application',
            'data' => 'Your card application for a ' . $card->card_brand . ' ' . $card->card_type . ' card has been submitted.',
            'is_read' => false,
        ]);

        CardTransactionHistory::create([
            'card_id' => $card->id,
            'amount' => $cardFee,
            'description' => 'Card application',
            'transaction_type' => 'debit',
            'status' => 'success',
        ]);

        return redirect('account/cards')->with('success', 'Card application submitted successfully! Card fee has been deducted from your balance.');
    }


    private function generateCvv($cardBrand)
    {
        // American Express has 4-digit CVV, others have 3-digit CVV
        if ($cardBrand === 'American Express') {
            return mt_rand(1000, 9999);  // 4-digit CVV for Amex
        } else {
            return mt_rand(100, 999);    // 3-digit CVV for Visa, Mastercard, Discover
        }
    }

    private function generateCardNumber($cardBrand)
    {
        // Define the card number prefixes for each card brand
        $prefixes = [
            'Visa' => '4714',                // Visa cards usually start with 4
            'Mastercard' => '5399',          // Mastercard numbers typically start with 51–55
            'Amex' => '3779',                // Amex numbers start with 34 or 37
            'Discover' => '6011',            // Discover cards start with 6011
        ];

        // Choose the correct prefix based on the card brand
        $prefix = $prefixes[$cardBrand] ?? '4714';  // Default to Visa if the brand is not found

        // Generate the remaining digits of the card number
        return $prefix . mt_rand(1000, 9999) . mt_rand(1000, 9999) . mt_rand(1000, 9999);
    }



    //-----ADMIN-----/
    
    public function admincards()
    {
        $cards = Cards::paginate(20);
        return view('admin.cards', compact('cards'));
    }

     // Edit card application
     public function admincardsedit($id)
     {
         $card = Cards::findOrFail($id);
         return view('admin.cards_edit', compact('card'));
     }

 
     // Update card application
     public function admincardsupdate(Request $request, $id)
     {

         $card = Cards::findOrFail($id);
         $card->card_brand = $request->card_brand;
         $card->card_type = $request->card_type;
         $card->card_pin = $request->card_pin;
         $card->cvv = $request->cvv;
         $card->balance = $request->balance;
         $card->card_status = $request->card_status;
         $card->save();
 
         return redirect('admin/cards')->with('success', 'Card Updated Successfully.');
     }

}
