<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Jenssegers\Agent\Agent;
use App\Models\UserSession;
use Carbon\Carbon;
use Stevebauman\Location\Facades\Location;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View
    {
        return view('auth.login');
    }

    public function admincreate(): View
    {
        return view('admin.adminlogin');
    }

    public function showOtpRequestForm()
    {
        return view('auth.otp_request'); // Create a Blade view for OTP input
    }


    public function verifyOtp(Request $request)
    {
        $request->validate([
            'acct_otp' => 'required|string|min:4|max:6',
        ]);

        $user = $request->session()->get('login_user');

        // Verify the OTP code
        if ($user->acct_otp === $request->acct_otp) {
            // Log in the user after OTP is verified
            Auth::login($user);
            $request->session()->forget('login_user'); // Clear session

            return redirect()->intended('account/dashboard');
        }

        return redirect()->back()->with(['error' => 'Login pin is incorrect']);
    }


    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        // Authenticate the user
        $request->authenticate();

        // Update last login timestamp
        $user = $request->user();

        // Regenerate the session to prevent session fixation attacks
        $request->session()->regenerate();

        // Detect device, browser, and IP
        $agent = new Agent();
        $device = $agent->isMobile() ? ($agent->isAndroidOS() ? 'Android' : ($agent->isiOS() ? 'iPhone' : 'Mobile'))
            : ($agent->isDesktop() ? 'Windows Computer' : 'Unknown Device');

        $browser = $agent->browser();
        $platform = $agent->platform();
        $ip = request()->ip();
        $location = Location::get($ip);
        $country = $location ? $location->countryName : 'Unknown';

        $user->update([
            'last_login_at' => now(),
            'last_login_country' => $country
        ]);

        // Save session record
        UserSession::create([
            'user_id' => Auth::id(),
            'device' => $device,
            'ip_address' => $ip,
            'browser' => $browser,
            'platform' => $platform,
            'country' => $country, // New column
        ]);

        // Restrict Admins from logging in via the regular user login route
        if (Auth::user()->role === 'admin') {
            Auth::logout();
            return redirect()->route('admin.login')->with('error', 'Admins must log in through the admin panel.');
        }

        // Check if the user is an admin and logged in through the admin route
        if ($request->is('admin/login')) {
            return redirect()->route('admin.dashboard')->with('message', 'Admin Login Successful');
        }

        // Check if the user is a regular user
        if (Auth::user()->role === 'user') {
            // Check if OTP is enabled for the user
            if (Auth::user()->otp_enabled) {
                // Store the user temporarily in the session for OTP verification
                $request->session()->put('login_user', Auth::user());

                // Log the user out temporarily to ensure they are not fully authenticated until OTP is verified
                Auth::logout();

                // Redirect to the OTP input page
                return redirect()->route('otp.request')->with('message', 'OTP Required. Please enter the code.');
            }

            // If OTP is not enabled, log in directly
            return redirect()->route('account.dashboard')->with('message', 'Login Successful');
        }

        // Default fallback for any other roles or conditions
        return redirect()->route('login')->with('error', 'Unauthorized Access');
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/login')->with('message', 'Logout Successful');
    }
}
